/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.josm.actions.search;

import java.io.PushbackReader;
import java.io.StringReader;
import java.text.Normalizer;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import org.openstreetmap.josm.Main;
import org.openstreetmap.josm.actions.search.PushbackTokenizer;
import org.openstreetmap.josm.data.Bounds;
import org.openstreetmap.josm.data.osm.Node;
import org.openstreetmap.josm.data.osm.OsmPrimitive;
import org.openstreetmap.josm.data.osm.OsmPrimitiveType;
import org.openstreetmap.josm.data.osm.OsmUtils;
import org.openstreetmap.josm.data.osm.Relation;
import org.openstreetmap.josm.data.osm.RelationMember;
import org.openstreetmap.josm.data.osm.Way;
import org.openstreetmap.josm.tools.Geometry;
import org.openstreetmap.josm.tools.I18n;
import org.openstreetmap.josm.tools.Predicate;
import org.openstreetmap.josm.tools.date.DateUtils;

public class SearchCompiler {
    private boolean caseSensitive = false;
    private boolean regexSearch = false;
    private static String rxErrorMsg = I18n.marktr("The regex \"{0}\" had a parse error at offset {1}, full error:\n\n{2}");
    private static String rxErrorMsgNoPos = I18n.marktr("The regex \"{0}\" had a parse error, full error:\n\n{1}");
    private PushbackTokenizer tokenizer;
    private static Map<String, SimpleMatchFactory> simpleMatchFactoryMap = new HashMap<String, SimpleMatchFactory>();
    private static Map<String, UnaryMatchFactory> unaryMatchFactoryMap = new HashMap<String, UnaryMatchFactory>();
    private static Map<String, BinaryMatchFactory> binaryMatchFactoryMap = new HashMap<String, BinaryMatchFactory>();

    public SearchCompiler(boolean caseSensitive, boolean regexSearch, PushbackTokenizer tokenizer) {
        this.caseSensitive = caseSensitive;
        this.regexSearch = regexSearch;
        this.tokenizer = tokenizer;
        if (simpleMatchFactoryMap.isEmpty()) {
            SearchCompiler.addMatchFactory(new CoreSimpleMatchFactory());
        }
        if (unaryMatchFactoryMap.isEmpty()) {
            SearchCompiler.addMatchFactory(new CoreUnaryMatchFactory());
        }
    }

    public static void addMatchFactory(MatchFactory factory) {
        for (String keyword : factory.getKeywords()) {
            if (factory instanceof SimpleMatchFactory) {
                simpleMatchFactoryMap.put(keyword, (SimpleMatchFactory)factory);
                continue;
            }
            if (factory instanceof UnaryMatchFactory) {
                unaryMatchFactoryMap.put(keyword, (UnaryMatchFactory)factory);
                continue;
            }
            if (factory instanceof BinaryMatchFactory) {
                binaryMatchFactoryMap.put(keyword, (BinaryMatchFactory)factory);
                continue;
            }
            throw new AssertionError((Object)"Unknown match factory");
        }
    }

    public static Match compile(String searchStr, boolean caseSensitive, boolean regexSearch) throws ParseError {
        return new SearchCompiler(caseSensitive, regexSearch, new PushbackTokenizer(new PushbackReader(new StringReader(searchStr)))).parse();
    }

    public Match parse() throws ParseError {
        Match m = this.parseExpression();
        if (!this.tokenizer.readIfEqual(PushbackTokenizer.Token.EOF)) {
            throw new ParseError(I18n.tr("Unexpected token: {0}", new Object[]{this.tokenizer.nextToken()}));
        }
        if (m == null) {
            return new Always();
        }
        return m;
    }

    private Match parseExpression() throws ParseError {
        Match factor = this.parseFactor();
        if (factor == null) {
            return null;
        }
        if (this.tokenizer.readIfEqual(PushbackTokenizer.Token.OR)) {
            return new Or(factor, this.parseExpression(I18n.tr("Missing parameter for OR", new Object[0])));
        }
        if (this.tokenizer.readIfEqual(PushbackTokenizer.Token.XOR)) {
            return new Xor(factor, this.parseExpression(I18n.tr("Missing parameter for XOR", new Object[0])));
        }
        Match expression = this.parseExpression();
        if (expression == null) {
            return factor;
        }
        return new And(factor, expression);
    }

    private Match parseExpression(String errorMessage) throws ParseError {
        Match expression = this.parseExpression();
        if (expression == null) {
            throw new ParseError(errorMessage);
        }
        return expression;
    }

    private Match parseFactor() throws ParseError {
        if (this.tokenizer.readIfEqual(PushbackTokenizer.Token.LEFT_PARENT)) {
            Match expression = this.parseExpression();
            if (!this.tokenizer.readIfEqual(PushbackTokenizer.Token.RIGHT_PARENT)) {
                throw new ParseError(PushbackTokenizer.Token.RIGHT_PARENT, this.tokenizer.nextToken());
            }
            return expression;
        }
        if (this.tokenizer.readIfEqual(PushbackTokenizer.Token.NOT)) {
            return new Not(this.parseFactor(I18n.tr("Missing operator for NOT", new Object[0])));
        }
        if (this.tokenizer.readIfEqual(PushbackTokenizer.Token.KEY)) {
            String key = this.tokenizer.getText();
            if (this.tokenizer.readIfEqual(PushbackTokenizer.Token.EQUALS)) {
                return new ExactKeyValue(this.regexSearch, key, this.tokenizer.readTextOrNumber());
            }
            if (this.tokenizer.readIfEqual(PushbackTokenizer.Token.LESS_THAN)) {
                return new ValueComparison(key, this.tokenizer.readTextOrNumber(), -1);
            }
            if (this.tokenizer.readIfEqual(PushbackTokenizer.Token.GREATER_THAN)) {
                return new ValueComparison(key, this.tokenizer.readTextOrNumber(), 1);
            }
            if (this.tokenizer.readIfEqual(PushbackTokenizer.Token.COLON)) {
                SimpleMatchFactory factory = simpleMatchFactoryMap.get(key);
                if (factory != null) {
                    return factory.get(key, this.tokenizer);
                }
                UnaryMatchFactory unaryFactory = unaryMatchFactoryMap.get(key);
                if (unaryFactory != null) {
                    return unaryFactory.get(key, this.parseFactor(), this.tokenizer);
                }
                return this.parseKV(key, this.tokenizer.readTextOrNumber());
            }
            if (this.tokenizer.readIfEqual(PushbackTokenizer.Token.QUESTION_MARK)) {
                return new BooleanMatch(key, false);
            }
            SimpleMatchFactory factory = simpleMatchFactoryMap.get(key);
            if (factory != null) {
                return factory.get(key, null);
            }
            UnaryMatchFactory unaryFactory = unaryMatchFactoryMap.get(key);
            if (unaryFactory != null) {
                return unaryFactory.get(key, this.parseFactor(), null);
            }
            return new Any(key, this.regexSearch, this.caseSensitive);
        }
        return null;
    }

    private Match parseFactor(String errorMessage) throws ParseError {
        Match fact = this.parseFactor();
        if (fact == null) {
            throw new ParseError(errorMessage);
        }
        return fact;
    }

    private Match parseKV(String key, String value) throws ParseError {
        if (value == null) {
            value = "";
        }
        switch (key) {
            case "type": {
                return new ExactType(value);
            }
            case "user": {
                return new UserMatch(value);
            }
            case "role": {
                return new RoleMatch(value);
            }
        }
        return new KeyValue(key, value, this.regexSearch, this.caseSensitive);
    }

    private static int regexFlags(boolean caseSensitive) {
        int searchFlags = 0;
        searchFlags |= 0x80;
        searchFlags |= 0x20;
        if (!caseSensitive) {
            searchFlags |= 0x42;
        }
        return searchFlags;
    }

    public static class ParseError
    extends Exception {
        public ParseError(String msg) {
            super(msg);
        }

        public ParseError(String msg, Throwable cause) {
            super(msg, cause);
        }

        public ParseError(PushbackTokenizer.Token expected, PushbackTokenizer.Token found) {
            this(I18n.tr("Unexpected token. Expected {0}, found {1}", new Object[]{expected, found}));
        }
    }

    private static class InView
    extends InArea {
        public InView(boolean all) {
            super(all);
        }

        @Override
        protected Bounds getBounds() {
            if (!Main.isDisplayingMapView()) {
                return null;
            }
            return Main.map.mapView.getRealBounds();
        }
    }

    private static class InDataSourceArea
    extends InArea {
        public InDataSourceArea(boolean all) {
            super(all);
        }

        @Override
        protected Bounds getBounds() {
            return new Bounds(Main.main.getCurrentDataSet().getDataSourceArea().getBounds2D());
        }
    }

    private static abstract class InArea
    extends Match {
        protected final boolean all;

        protected abstract Bounds getBounds();

        public InArea(boolean all) {
            this.all = all;
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            if (!osm.isUsable()) {
                return false;
            }
            if (osm instanceof Node) {
                Bounds bounds = this.getBounds();
                return bounds != null && bounds.contains(((Node)osm).getCoor());
            }
            if (osm instanceof Way) {
                List<Node> nodes = ((Way)osm).getNodes();
                return this.all ? this.forallMatch(nodes) : this.existsMatch(nodes);
            }
            if (osm instanceof Relation) {
                Set<OsmPrimitive> primitives = ((Relation)osm).getMemberPrimitives();
                return this.all ? this.forallMatch(primitives) : this.existsMatch(primitives);
            }
            return false;
        }
    }

    private static class AreaSize
    extends RangeMatch {
        public AreaSize(PushbackTokenizer.Range range) {
            super(range);
        }

        public AreaSize(PushbackTokenizer tokenizer) throws ParseError {
            this(tokenizer.readRange(I18n.tr("Range of numbers expected", new Object[0])));
        }

        @Override
        protected Long getNumber(OsmPrimitive osm) {
            if (!(osm instanceof Way) || !((Way)osm).isClosed()) {
                return null;
            }
            Way way = (Way)osm;
            return (long)Geometry.closedWayArea(way);
        }

        @Override
        protected String getString() {
            return "areasize";
        }
    }

    public static class Child
    extends UnaryMatch {
        public Child(Match m) {
            super(m);
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            boolean isChild = false;
            for (OsmPrimitive p : osm.getReferrers()) {
                isChild |= this.match.match(p);
            }
            return isChild;
        }

        public String toString() {
            return "child(" + this.match + ")";
        }
    }

    public static class Parent
    extends UnaryMatch {
        public Parent(Match m) {
            super(m);
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            boolean isParent;
            block3: {
                block2: {
                    isParent = false;
                    if (!(osm instanceof Way)) break block2;
                    for (Node n : ((Way)osm).getNodes()) {
                        isParent |= this.match.match(n);
                    }
                    break block3;
                }
                if (!(osm instanceof Relation)) break block3;
                for (RelationMember member : ((Relation)osm).getMembers()) {
                    isParent |= this.match.match(member.getMember());
                }
            }
            return isParent;
        }

        public String toString() {
            return "parent(" + this.match + ")";
        }
    }

    private static class Closed
    extends Match {
        private Closed() {
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            return osm instanceof Way && ((Way)osm).isClosed();
        }

        public String toString() {
            return "closed";
        }
    }

    private static class Untagged
    extends Match {
        private Untagged() {
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            return !osm.isTagged() && !osm.isIncomplete();
        }

        public String toString() {
            return "untagged";
        }
    }

    private static class Incomplete
    extends Match {
        private Incomplete() {
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            return osm.isIncomplete();
        }

        public String toString() {
            return "incomplete";
        }
    }

    private static class Selected
    extends Match {
        private Selected() {
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            return Main.main.getCurrentDataSet().isSelected(osm);
        }

        public String toString() {
            return "selected";
        }
    }

    private static class Modified
    extends Match {
        private Modified() {
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            return osm.isModified() || osm.isNewOrUndeleted();
        }

        public String toString() {
            return "modified";
        }
    }

    private static class New
    extends Match {
        private New() {
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            return osm.isNew();
        }

        public String toString() {
            return "new";
        }
    }

    private static class TimestampRange
    extends RangeMatch {
        public TimestampRange(long minCount, long maxCount) {
            super(minCount, maxCount);
        }

        @Override
        protected Long getNumber(OsmPrimitive osm) {
            return osm.getTimestamp().getTime();
        }

        @Override
        protected String getString() {
            return "timestamp";
        }
    }

    private static class TagCountRange
    extends RangeMatch {
        public TagCountRange(PushbackTokenizer.Range range) {
            super(range);
        }

        public TagCountRange(PushbackTokenizer tokenizer) throws ParseError {
            this(tokenizer.readRange(I18n.tr("Range of numbers expected", new Object[0])));
        }

        @Override
        protected Long getNumber(OsmPrimitive osm) {
            return osm.getKeys().size();
        }

        @Override
        protected String getString() {
            return "tags";
        }
    }

    private static class NodeCountRange
    extends RangeMatch {
        public NodeCountRange(PushbackTokenizer.Range range) {
            super(range);
        }

        public NodeCountRange(PushbackTokenizer tokenizer) throws ParseError {
            this(tokenizer.readRange(I18n.tr("Range of numbers expected", new Object[0])));
        }

        @Override
        protected Long getNumber(OsmPrimitive osm) {
            if (!(osm instanceof Way)) {
                return null;
            }
            return ((Way)osm).getRealNodesCount();
        }

        @Override
        protected String getString() {
            return "nodes";
        }
    }

    private static abstract class RangeMatch
    extends Match {
        private final long min;
        private final long max;

        public RangeMatch(long min, long max) {
            this.min = Math.min(min, max);
            this.max = Math.max(min, max);
        }

        public RangeMatch(PushbackTokenizer.Range range) {
            this(range.getStart(), range.getEnd());
        }

        protected abstract Long getNumber(OsmPrimitive var1);

        protected abstract String getString();

        @Override
        public boolean match(OsmPrimitive osm) {
            Long num = this.getNumber(osm);
            if (num == null) {
                return false;
            }
            return num >= this.min && num <= this.max;
        }

        public String toString() {
            return this.getString() + "=" + this.min + "-" + this.max;
        }
    }

    private static class Nth
    extends Match {
        private final int nth;
        private final boolean modulo;

        public Nth(PushbackTokenizer tokenizer, boolean modulo) throws ParseError {
            this((int)tokenizer.readNumber(I18n.tr("Positive integer expected", new Object[0])), modulo);
        }

        private Nth(int nth, boolean modulo) throws ParseError {
            if (nth <= 0) {
                throw new ParseError(I18n.tr("Positive integer expected", new Object[0]));
            }
            this.nth = nth;
            this.modulo = modulo;
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            for (OsmPrimitive p : osm.getReferrers()) {
                Integer idx = null;
                if (p instanceof Way) {
                    Way w = (Way)p;
                    idx = w.getNodes().indexOf(osm);
                } else if (p instanceof Relation) {
                    Relation r = (Relation)p;
                    idx = r.getMemberPrimitivesList().indexOf(osm);
                }
                if (idx == null || idx != this.nth && (!this.modulo || idx % this.nth != 0)) continue;
                return true;
            }
            return false;
        }
    }

    private static class RoleMatch
    extends Match {
        private String role;

        public RoleMatch(String role) {
            this.role = role == null ? "" : role;
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            for (OsmPrimitive ref : osm.getReferrers()) {
                if (!(ref instanceof Relation) || ref.isIncomplete() || ref.isDeleted()) continue;
                for (RelationMember m : ((Relation)ref).getMembers()) {
                    String testRole;
                    if (m.getMember() != osm || !this.role.equals((testRole = m.getRole()) == null ? "" : testRole)) continue;
                    return true;
                }
            }
            return false;
        }

        public String toString() {
            return "role=" + this.role;
        }
    }

    private static class UserMatch
    extends Match {
        private String user;

        public UserMatch(String user) {
            this.user = "anonymous".equals(user) ? null : user;
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            if (osm.getUser() == null) {
                return this.user == null;
            }
            return osm.getUser().hasName(this.user);
        }

        public String toString() {
            return "user=" + (this.user == null ? "" : this.user);
        }
    }

    private static class ExactType
    extends Match {
        private final OsmPrimitiveType type;

        public ExactType(String type) throws ParseError {
            this.type = OsmPrimitiveType.from(type);
            if (this.type == null) {
                throw new ParseError(I18n.tr("Unknown primitive type: {0}. Allowed values are node, way or relation", type));
            }
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            return this.type.equals((Object)osm.getType());
        }

        public String toString() {
            return "type=" + (Object)((Object)this.type);
        }
    }

    private static class Any
    extends Match {
        private final String search;
        private final Pattern searchRegex;
        private final boolean caseSensitive;

        public Any(String s, boolean regexSearch, boolean caseSensitive) throws ParseError {
            s = Normalizer.normalize(s, Normalizer.Form.NFC);
            this.caseSensitive = caseSensitive;
            if (regexSearch) {
                try {
                    this.searchRegex = Pattern.compile(s, SearchCompiler.regexFlags(caseSensitive));
                }
                catch (PatternSyntaxException e) {
                    throw new ParseError(I18n.tr(rxErrorMsg, e.getPattern(), e.getIndex(), e.getMessage()), e);
                }
                catch (Exception e) {
                    throw new ParseError(I18n.tr(rxErrorMsgNoPos, s, e.getMessage()), e);
                }
                this.search = s;
            } else if (caseSensitive) {
                this.search = s;
                this.searchRegex = null;
            } else {
                this.search = s.toLowerCase();
                this.searchRegex = null;
            }
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            if (!osm.hasKeys() && osm.getUser() == null) {
                return this.search.isEmpty();
            }
            for (String key : osm.keySet()) {
                String value = osm.get(key);
                if (this.searchRegex != null) {
                    value = Normalizer.normalize(value, Normalizer.Form.NFC);
                    Matcher keyMatcher = this.searchRegex.matcher(key);
                    Matcher valMatcher = this.searchRegex.matcher(value);
                    boolean keyMatchFound = keyMatcher.find();
                    boolean valMatchFound = valMatcher.find();
                    if (!keyMatchFound && !valMatchFound) continue;
                    return true;
                }
                if (!this.caseSensitive) {
                    key = key.toLowerCase();
                    value = value.toLowerCase();
                }
                value = Normalizer.normalize(value, Normalizer.Form.NFC);
                if (key.indexOf(this.search) == -1 && value.indexOf(this.search) == -1) continue;
                return true;
            }
            return false;
        }

        public String toString() {
            return this.search;
        }
    }

    public static class ExactKeyValue
    extends Match {
        private final String key;
        private final String value;
        private final Pattern keyPattern;
        private final Pattern valuePattern;
        private final Mode mode;

        public ExactKeyValue(boolean regexp, String key, String value) throws ParseError {
            if ("".equals(key)) {
                throw new ParseError(I18n.tr("Key cannot be empty when tag operator is used. Sample use: key=value", new Object[0]));
            }
            this.key = key;
            String string = this.value = value == null ? "" : value;
            this.mode = "".equals(this.value) && "*".equals(key) ? Mode.NONE : ("".equals(this.value) ? (regexp ? Mode.MISSING_KEY_REGEXP : Mode.MISSING_KEY) : ("*".equals(key) && "*".equals(this.value) ? Mode.ANY : ("*".equals(key) ? (regexp ? Mode.ANY_KEY_REGEXP : Mode.ANY_KEY) : ("*".equals(this.value) ? (regexp ? Mode.ANY_VALUE_REGEXP : Mode.ANY_VALUE) : (regexp ? Mode.EXACT_REGEXP : Mode.EXACT)))));
            if (regexp && key.length() > 0 && !"*".equals(key)) {
                try {
                    this.keyPattern = Pattern.compile(key, SearchCompiler.regexFlags(false));
                }
                catch (PatternSyntaxException e) {
                    throw new ParseError(I18n.tr(rxErrorMsg, e.getPattern(), e.getIndex(), e.getMessage()));
                }
                catch (Exception e) {
                    throw new ParseError(I18n.tr(rxErrorMsgNoPos, key, e.getMessage()));
                }
            } else {
                this.keyPattern = null;
            }
            if (regexp && this.value.length() > 0 && !"*".equals(this.value)) {
                try {
                    this.valuePattern = Pattern.compile(this.value, SearchCompiler.regexFlags(false));
                }
                catch (PatternSyntaxException e) {
                    throw new ParseError(I18n.tr(rxErrorMsg, e.getPattern(), e.getIndex(), e.getMessage()));
                }
                catch (Exception e) {
                    throw new ParseError(I18n.tr(rxErrorMsgNoPos, value, e.getMessage()));
                }
            } else {
                this.valuePattern = null;
            }
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            if (!osm.hasKeys()) {
                return this.mode == Mode.NONE;
            }
            switch (this.mode) {
                case NONE: {
                    return false;
                }
                case MISSING_KEY: {
                    return osm.get(this.key) == null;
                }
                case ANY: {
                    return true;
                }
                case ANY_VALUE: {
                    return osm.get(this.key) != null;
                }
                case ANY_KEY: {
                    for (String v : osm.getKeys().values()) {
                        if (!v.equals(this.value)) continue;
                        return true;
                    }
                    return false;
                }
                case EXACT: {
                    return this.value.equals(osm.get(this.key));
                }
                case ANY_KEY_REGEXP: {
                    for (String v : osm.getKeys().values()) {
                        if (!this.valuePattern.matcher(v).matches()) continue;
                        return true;
                    }
                    return false;
                }
                case ANY_VALUE_REGEXP: 
                case EXACT_REGEXP: {
                    for (String key : osm.keySet()) {
                        if (!this.keyPattern.matcher(key).matches() || this.mode != Mode.ANY_VALUE_REGEXP && !this.valuePattern.matcher(osm.get(key)).matches()) continue;
                        return true;
                    }
                    return false;
                }
                case MISSING_KEY_REGEXP: {
                    for (String k : osm.keySet()) {
                        if (!this.keyPattern.matcher(k).matches()) continue;
                        return false;
                    }
                    return true;
                }
            }
            throw new AssertionError((Object)"Missed state");
        }

        public String toString() {
            return this.key + '=' + this.value;
        }

        private static enum Mode {
            ANY,
            ANY_KEY,
            ANY_VALUE,
            EXACT,
            NONE,
            MISSING_KEY,
            ANY_KEY_REGEXP,
            ANY_VALUE_REGEXP,
            EXACT_REGEXP,
            MISSING_KEY_REGEXP;

        }
    }

    public static class ValueComparison
    extends Match {
        private final String key;
        private final String referenceValue;
        private final int compareMode;

        public ValueComparison(String key, String referenceValue, int compareMode) {
            this.key = key;
            this.referenceValue = referenceValue;
            this.compareMode = compareMode;
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            int compareResult;
            String currentValue = osm.get(this.key);
            if (currentValue == null) {
                return false;
            }
            try {
                compareResult = Double.compare(Double.parseDouble(currentValue), Double.parseDouble(this.referenceValue));
            }
            catch (NumberFormatException ignore) {
                compareResult = osm.get(this.key).compareTo(this.referenceValue);
            }
            return this.compareMode < 0 ? compareResult < 0 : (this.compareMode > 0 ? compareResult > 0 : compareResult == 0);
        }
    }

    private static class KeyValue
    extends Match {
        private final String key;
        private final Pattern keyPattern;
        private final String value;
        private final Pattern valuePattern;
        private final boolean caseSensitive;

        public KeyValue(String key, String value, boolean regexSearch, boolean caseSensitive) throws ParseError {
            this.caseSensitive = caseSensitive;
            if (regexSearch) {
                int searchFlags = SearchCompiler.regexFlags(caseSensitive);
                try {
                    this.keyPattern = Pattern.compile(key, searchFlags);
                }
                catch (PatternSyntaxException e) {
                    throw new ParseError(I18n.tr(rxErrorMsg, e.getPattern(), e.getIndex(), e.getMessage()), e);
                }
                catch (Exception e) {
                    throw new ParseError(I18n.tr(rxErrorMsgNoPos, key, e.getMessage()), e);
                }
                try {
                    this.valuePattern = Pattern.compile(value, searchFlags);
                }
                catch (PatternSyntaxException e) {
                    throw new ParseError(I18n.tr(rxErrorMsg, e.getPattern(), e.getIndex(), e.getMessage()), e);
                }
                catch (Exception e) {
                    throw new ParseError(I18n.tr(rxErrorMsgNoPos, value, e.getMessage()), e);
                }
                this.key = key;
                this.value = value;
            } else if (caseSensitive) {
                this.key = key;
                this.value = value;
                this.keyPattern = null;
                this.valuePattern = null;
            } else {
                this.key = key.toLowerCase();
                this.value = value;
                this.keyPattern = null;
                this.valuePattern = null;
            }
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            if (this.keyPattern != null) {
                if (!osm.hasKeys()) {
                    return false;
                }
                for (String k : osm.keySet()) {
                    Matcher matcherValue;
                    boolean matchedValue;
                    String v = osm.get(k);
                    Matcher matcherKey = this.keyPattern.matcher(k);
                    boolean matchedKey = matcherKey.find();
                    if (!matchedKey || !(matchedValue = (matcherValue = this.valuePattern.matcher(v)).find())) continue;
                    return true;
                }
            } else {
                String mv = null;
                mv = "timestamp".equals(this.key) ? DateUtils.fromDate(osm.getTimestamp()) : osm.get(this.key);
                if (mv == null) {
                    return false;
                }
                String v1 = this.caseSensitive ? mv : mv.toLowerCase();
                String v2 = this.caseSensitive ? this.value : this.value.toLowerCase();
                v1 = Normalizer.normalize(v1, Normalizer.Form.NFC);
                v2 = Normalizer.normalize(v2, Normalizer.Form.NFC);
                return v1.indexOf(v2) != -1;
            }
            return false;
        }

        public String toString() {
            return this.key + "=" + this.value;
        }
    }

    private static class Version
    extends RangeMatch {
        public Version(PushbackTokenizer.Range range) {
            super(range);
        }

        public Version(PushbackTokenizer tokenizer) throws ParseError {
            this(tokenizer.readRange(I18n.tr("Range of versions expected", new Object[0])));
        }

        @Override
        protected Long getNumber(OsmPrimitive osm) {
            return osm.getVersion();
        }

        @Override
        protected String getString() {
            return "version";
        }
    }

    private static class ChangesetId
    extends RangeMatch {
        public ChangesetId(PushbackTokenizer.Range range) {
            super(range);
        }

        public ChangesetId(PushbackTokenizer tokenizer) throws ParseError {
            this(tokenizer.readRange(I18n.tr("Range of changeset ids expected", new Object[0])));
        }

        @Override
        protected Long getNumber(OsmPrimitive osm) {
            return osm.getChangesetId();
        }

        @Override
        protected String getString() {
            return "changeset";
        }
    }

    private static class Id
    extends RangeMatch {
        public Id(PushbackTokenizer.Range range) {
            super(range);
        }

        public Id(PushbackTokenizer tokenizer) throws ParseError {
            this(tokenizer.readRange(I18n.tr("Range of primitive ids expected", new Object[0])));
        }

        @Override
        protected Long getNumber(OsmPrimitive osm) {
            return osm.isNew() ? 0L : osm.getUniqueId();
        }

        @Override
        protected String getString() {
            return "id";
        }
    }

    public static class Xor
    extends BinaryMatch {
        public Xor(Match lhs, Match rhs) {
            super(lhs, rhs);
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            return this.lhs.match(osm) ^ this.rhs.match(osm);
        }

        public String toString() {
            return this.lhs + " ^ " + this.rhs;
        }
    }

    public static class Or
    extends BinaryMatch {
        public Or(Match lhs, Match rhs) {
            super(lhs, rhs);
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            return this.lhs.match(osm) || this.rhs.match(osm);
        }

        public String toString() {
            return this.lhs + " || " + this.rhs;
        }
    }

    public static class And
    extends BinaryMatch {
        public And(Match lhs, Match rhs) {
            super(lhs, rhs);
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            return this.lhs.match(osm) && this.rhs.match(osm);
        }

        public String toString() {
            return this.lhs + " && " + this.rhs;
        }
    }

    private static class BooleanMatch
    extends Match {
        private final String key;
        private final boolean defaultValue;

        public BooleanMatch(String key, boolean defaultValue) {
            this.key = key;
            this.defaultValue = defaultValue;
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            Boolean ret = OsmUtils.getOsmBoolean(osm.get(this.key));
            if (ret == null) {
                return this.defaultValue;
            }
            return ret;
        }
    }

    public static class Not
    extends UnaryMatch {
        public Not(Match match) {
            super(match);
        }

        @Override
        public boolean match(OsmPrimitive osm) {
            return !this.match.match(osm);
        }

        public String toString() {
            return "!" + this.match;
        }

        public Match getMatch() {
            return this.match;
        }
    }

    public static class Never
    extends Match {
        @Override
        public boolean match(OsmPrimitive osm) {
            return false;
        }
    }

    public static class Always
    extends Match {
        public static final Always INSTANCE = new Always();

        @Override
        public boolean match(OsmPrimitive osm) {
            return true;
        }
    }

    public static abstract class BinaryMatch
    extends Match {
        protected final Match lhs;
        protected final Match rhs;

        public BinaryMatch(Match lhs, Match rhs) {
            this.lhs = lhs;
            this.rhs = rhs;
        }

        public Match getLhs() {
            return this.lhs;
        }

        public Match getRhs() {
            return this.rhs;
        }
    }

    public static abstract class UnaryMatch
    extends Match {
        protected final Match match;

        public UnaryMatch(Match match) {
            this.match = match == null ? new Always() : match;
        }

        public Match getOperand() {
            return this.match;
        }
    }

    public static abstract class Match
    implements Predicate<OsmPrimitive> {
        public abstract boolean match(OsmPrimitive var1);

        protected boolean existsMatch(Collection<? extends OsmPrimitive> primitives) {
            for (OsmPrimitive osmPrimitive : primitives) {
                if (!this.match(osmPrimitive)) continue;
                return true;
            }
            return false;
        }

        protected boolean forallMatch(Collection<? extends OsmPrimitive> primitives) {
            for (OsmPrimitive osmPrimitive : primitives) {
                if (this.match(osmPrimitive)) continue;
                return false;
            }
            return true;
        }

        @Override
        public final boolean evaluate(OsmPrimitive object) {
            return this.match(object);
        }
    }

    public static interface BinaryMatchFactory
    extends MatchFactory {
        public BinaryMatch get(String var1, Match var2, Match var3, PushbackTokenizer var4) throws ParseError;
    }

    public static interface UnaryMatchFactory
    extends MatchFactory {
        public UnaryMatch get(String var1, Match var2, PushbackTokenizer var3) throws ParseError;
    }

    public static interface SimpleMatchFactory
    extends MatchFactory {
        public Match get(String var1, PushbackTokenizer var2) throws ParseError;
    }

    private static interface MatchFactory {
        public Collection<String> getKeywords();
    }

    public static class CoreUnaryMatchFactory
    implements UnaryMatchFactory {
        private static Collection<String> keywords = Arrays.asList("parent", "child");

        @Override
        public UnaryMatch get(String keyword, Match matchOperand, PushbackTokenizer tokenizer) {
            if ("parent".equals(keyword)) {
                return new Parent(matchOperand);
            }
            if ("child".equals(keyword)) {
                return new Child(matchOperand);
            }
            return null;
        }

        @Override
        public Collection<String> getKeywords() {
            return keywords;
        }
    }

    public class CoreSimpleMatchFactory
    implements SimpleMatchFactory {
        private Collection<String> keywords = Arrays.asList("id", "version", "changeset", "nodes", "tags", "areasize", "modified", "selected", "incomplete", "untagged", "closed", "new", "indownloadedarea", "allindownloadedarea", "inview", "allinview", "timestamp", "nth", "nth%");

        @Override
        public Match get(String keyword, PushbackTokenizer tokenizer) throws ParseError {
            switch (keyword) {
                case "modified": {
                    return new Modified();
                }
                case "selected": {
                    return new Selected();
                }
                case "incomplete": {
                    return new Incomplete();
                }
                case "untagged": {
                    return new Untagged();
                }
                case "closed": {
                    return new Closed();
                }
                case "new": {
                    return new New();
                }
                case "indownloadedarea": {
                    return new InDataSourceArea(false);
                }
                case "allindownloadedarea": {
                    return new InDataSourceArea(true);
                }
                case "inview": {
                    return new InView(false);
                }
                case "allinview": {
                    return new InView(true);
                }
            }
            if (tokenizer != null) {
                switch (keyword) {
                    case "id": {
                        return new Id(tokenizer);
                    }
                    case "version": {
                        return new Version(tokenizer);
                    }
                    case "changeset": {
                        return new ChangesetId(tokenizer);
                    }
                    case "nodes": {
                        return new NodeCountRange(tokenizer);
                    }
                    case "tags": {
                        return new TagCountRange(tokenizer);
                    }
                    case "areasize": {
                        return new AreaSize(tokenizer);
                    }
                    case "nth": {
                        return new Nth(tokenizer, false);
                    }
                    case "nth%": {
                        return new Nth(tokenizer, true);
                    }
                    case "timestamp": {
                        String rangeS = " " + tokenizer.readTextOrNumber() + " ";
                        String[] rangeA = rangeS.split("/");
                        if (rangeA.length == 1) {
                            return new KeyValue(keyword, rangeS.trim(), SearchCompiler.this.regexSearch, SearchCompiler.this.caseSensitive);
                        }
                        if (rangeA.length == 2) {
                            String rangeA1 = rangeA[0].trim();
                            String rangeA2 = rangeA[1].trim();
                            long minDate = DateUtils.fromString(rangeA1.isEmpty() ? "1980" : rangeA1).getTime();
                            long maxDate = rangeA2.isEmpty() ? System.currentTimeMillis() : DateUtils.fromString(rangeA2).getTime();
                            return new TimestampRange(minDate, maxDate);
                        }
                        throw new ParseError(I18n.tr("Expecting <i>min</i>/<i>max</i> after ''timestamp''", new Object[0]));
                    }
                }
            }
            return null;
        }

        @Override
        public Collection<String> getKeywords() {
            return this.keywords;
        }
    }
}

